------------------------------------------------------------------------------
--                                                                          --
--                           GNATTEST COMPONENTS                            --
--                                                                          --
--                      G N A T T E S T . C O M M O N                       --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                       Copyright (C) 2011, AdaCore                        --
--                                                                          --
-- GNATTEST  is  free  software;  you  can redistribute it and/or modify it --
-- under terms of the  GNU  General Public License as published by the Free --
-- Software  Foundation;  either  version  2, or (at your option) any later --
-- version.  GNATTEST  is  distributed  in the hope that it will be useful, --
-- but  WITHOUT  ANY  WARRANTY;   without  even  the  implied  warranty  of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General --
-- Public License for more details.  You should have received a copy of the --
-- GNU  General  Public License distributed with GNAT; see file COPYING. If --
-- not, write to the  Free  Software  Foundation, 51 Franklin Street, Fifth --
-- Floor, Boston, MA 02110-1301, USA.,                                      --
--                                                                          --
-- GNATTEST is maintained by AdaCore (http://www.adacore.com).              --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Characters.Handling;     use Ada.Characters.Handling;

with Asis.Implementation;         use Asis.Implementation;

with GNAT.Directory_Operations;   use GNAT.Directory_Operations;

with GNATtest.Options;            use GNATtest.Options;

package body GNATtest.Common is

   --------------------------
   -- Generate_Common_File --
   --------------------------

   procedure Generate_Common_File is
      Common_Package_Name : constant String := "Gnattest_Generated";
      Common_File_Subdir  : constant String :=
        Harness_Dir.all & Directory_Separator & "common";
   begin
      if not Is_Directory (Common_File_Subdir) then
         Make_Dir (Common_File_Subdir);
      end if;
      Create (Output_File,
              Out_File,
              Common_File_Subdir &
              Directory_Separator &
              Unit_To_File_Name (Common_Package_Name) & ".ads");

      S_Put (0, "package Gnattest_Generated is");
      New_Line (Output_File);
      S_Put (3, "package GNATtest_Standard renames Standard;");
      New_Line (Output_File);
      S_Put (3, "Default_Assert_Value : Boolean := ");
      if Stubs_Fail then
         S_Put (0, "False;");
      else
         S_Put (0, "True;");
      end if;
      New_Line (Output_File);
      S_Put (0, "end Gnattest_Generated;");

      Close (Output_File);
   end Generate_Common_File;

   ------------------------
   -- Report_AUnit_Usage --
   ------------------------

   procedure Report_AUnit_Usage is
   begin
      Report_Err ("gnattest: trying to process aunit itself!");
      Report_Err ("gnattest: Fatal_Error raised, terminating process.");
   end Report_AUnit_Usage;

   ----------------
   -- Report_Err --
   ----------------

   procedure Report_Err (Message : String) is
   begin
      Ada.Text_IO.Put_Line (Ada.Text_IO.Standard_Error, Message);
   end Report_Err;

   ----------------
   -- Report_Std --
   ----------------

   procedure Report_Std (Message : String; Offset : Integer := 0) is
   begin

      if GNATtest.Options.Quiet then
         return;
      end if;

      for I in 1 .. Offset loop
         Ada.Text_IO.Put (Ada.Text_IO.Standard_Output, " ");
      end loop;

      Ada.Text_IO.Put_Line (Ada.Text_IO.Standard_Output, Message);
   end Report_Std;

   -------------------------------------
   -- Report_Unhandled_ASIS_Exception --
   -------------------------------------

   procedure Report_Unhandled_ASIS_Exception (Ex : Exception_Occurrence) is
   begin
      Report_Err ("ASIS exception (" & Exception_Name (Ex) & ") is raised");
      Report_Err ("ASIS Error Status is " & Status'Img);
      Report_Err ("ASIS Diagnosis is " & To_String (Diagnosis));

      Set_Status;
   end Report_Unhandled_ASIS_Exception;

   --------------------------------
   -- Report_Unhandled_Exception --
   --------------------------------

   procedure Report_Unhandled_Exception (Ex : Exception_Occurrence) is
   begin
      Report_Err (Exception_Information (Ex));
   end Report_Unhandled_Exception;

   -----------
   -- S_Put --
   -----------

   procedure S_Put (Span : Natural; Text : String) is
   begin
      for J in 0 .. Span - 1 loop
         Put (Output_File, " ");
      end loop;
      Put (Output_File, Text);
   end S_Put;

   -----------------------
   -- Unit_To_File_Name --
   -----------------------

   function Unit_To_File_Name (Old : String) return String is
      T : String_Access;
   begin
      T := new String'(Old);
      for J in T.all'First .. T.all'Last loop
         if T.all (J) = '.' then
            T.all (J) := '-';
         end if;
      end loop;

      return To_Lower (T.all);
   end Unit_To_File_Name;

end GNATtest.Common;
